unit Formula;

interface

uses
  SysUtils,
  Classes,
  StreamSec.DSI.Arith.OO;

type
  tAbstractExpression = class
  public
    procedure Parse(const aExpression: string); virtual; abstract;
    function Evaluate: tDSIntRec; virtual; abstract;
    procedure GetVariableNames(aList: TStrings); virtual; abstract;
    procedure SetVariables(aList: TStrings); virtual; abstract;
  end;

  tValue = class(tAbstractExpression)
  private
    fValue: tDSIntRec;
  public
    constructor Create(const aValue: string);
    procedure Parse(const aExpression: string); override;
    function Evaluate: tDSIntRec; override;
    procedure GetVariableNames(aList: TStrings); override;
    procedure SetVariables(aList: TStrings); override;
  end;

  tVariable = class(tValue)
  private
    fName: string;
    procedure SetName(const Value: string);
  public
    constructor Create(const aName: string);
    procedure GetVariableNames(aList: TStrings); override;
    procedure SetVariables(aList: TStrings); override;
    property Name: string read fName write SetName;
  end;

  tExpression = class(tAbstractExpression)
  private
    fLeft: tAbstractExpression;
    fRight: tAbstractExpression;
    fOperator: string;
  public
    constructor Create(const aExpression: string);
    destructor Destroy; override;
    procedure Parse(const aExpression: string); override;
    function Evaluate: tDSIntRec; override;
    procedure GetVariableNames(aList: TStrings); override;
    procedure SetVariables(aList: TStrings); override;
  end;

implementation

{ tValue }

constructor tValue.Create(const aValue: string);
begin
  fValue := tDSIntRec(aValue);
end;

function tValue.Evaluate: tDSIntRec;
begin
  Result := fValue;
end;

procedure tValue.GetVariableNames(aList: TStrings);
begin
  // do nothing
end;

procedure tValue.Parse(const aExpression: string);
begin
  //
end;

procedure tValue.SetVariables(aList: TStrings);
begin
  //
end;

{ tVariable }

constructor tVariable.Create(const aName: string);
begin
  fName := aName;
end;

procedure tVariable.GetVariableNames(aList: TStrings);
begin
  if aList.IndexOf(fName) < 0 then
    aList.Add(fName);
end;

procedure tVariable.SetName(const Value: string);
begin
  fName := Value;
end;

procedure tVariable.SetVariables(aList: TStrings);
begin
  fValue := tDSIntRec(aList.Values[fName]);
end;

{ tExpression }

constructor tExpression.Create(const aExpression: string);
begin
  Parse(aExpression);
end;

destructor tExpression.Destroy;
begin
  fLeft.Free;
  fRight.Free;
  inherited;
end;

function tExpression.Evaluate: tDSIntRec;
begin
  if fOperator = '' then
    Result := fLeft.Evaluate
  else if fOperator = '+' then
    Result := fLeft.Evaluate + fRight.Evaluate
  else if fOperator = '-' then
    Result := fLeft.Evaluate - fRight.Evaluate
  else if fOperator = '*' then
    Result := fLeft.Evaluate * fRight.Evaluate
  else if fOperator = '\' then
    Result := fLeft.Evaluate div fRight.Evaluate
  else if SameText(fOperator,'|') then
    Result := fLeft.Evaluate mod fRight.Evaluate
  else if fOperator = '%' then begin
    Result := fLeft.Evaluate mod fRight.Evaluate;
    Result.fMod := nil;
  end else if fOperator = '<' then
    Result := fLeft.Evaluate shl Integer(fRight.Evaluate)
  else if fOperator = '>' then
    Result := fLeft.Evaluate shr Integer(fRight.Evaluate)
  else if fOperator = '/' then
    Result := fLeft.Evaluate / fRight.Evaluate
  else if fOperator = '^' then
    Result := ExpMod(fLeft.Evaluate,fRight.Evaluate)
  else if SameText(fOperator,'GCF') or SameText(fOperator,'GCD') then begin
    Result := GCD(tExpression(fLeft).fLeft.Evaluate,tExpression(fLeft).fRight.Evaluate)
  end else if SameText(fOperator,'LCM') or SameText(fOperator,'LCD') then begin
    Result := LCD(tExpression(fLeft).fLeft.Evaluate,tExpression(fLeft).fRight.Evaluate)
  end else if SameText(fOperator,'MSB') then
    Result := fLeft.Evaluate.MSB
  else if SameText(fOperator,'Random') or SameText(fOperator,'Rnd') or SameText(fOperator,'Rand') then
    Result.Random(Integer(fLeft.Evaluate))
  else if SameText(fOperator,'Prime') then
    Result := Prime(Integer(fLeft.Evaluate))
  else
    raise Exception.Create('Unknown operator: ' + fOperator);
end;

procedure tExpression.GetVariableNames(aList: TStrings);
begin
  if Assigned(fLeft) then
    fLeft.GetVariableNames(aList);
  if Assigned(fRight) then
    fRight.GetVariableNames(aList);
end;

procedure tExpression.Parse(const aExpression: string);
type
  tPos = (pBlank,pExpr,pOp);
var
  lPos: tPos;
  lIdx: Integer;
  lStart: Integer;
  lLevel: Integer;
  lSL: TStringList;
  lIdx2: Integer;
  lExpr: string;
begin
  FreeAndNil(fLeft);
  FreeAndNil(fRight);
  fOperator := '';
  lStart := 0;
  lIdx := 1;
  lLevel := 0;
  lPos := pBlank;
  lSL := TStringList.Create;
  try
    while lIdx <= Length(aExpression) do begin
      case aExpression[lIdx] of
        '(':
          begin
            if lLevel = 0 then begin
              if lPos = pExpr then
                lSL.Add(Copy(aExpression,lStart,lIdx-lStart));
              lPos := pExpr;
              lStart := lIdx;
            end;
            Inc(lLevel);
          end;
        ')':
          begin
            Dec(lLevel);
            if lLevel = 0 then begin
              lSL.AddObject(Copy(aExpression,lStart+1,lIdx-lStart-1),TObject(1));
              lPos := pBlank;
            end else if lLevel < 0 then
              raise Exception.Create('Syntax error');
          end;
        #0..#32:
          if (lLevel = 0) and (lPos <> pBlank) then begin
            if lPos = pExpr then
              lSL.Add(Copy(aExpression,lStart,lIdx-lStart));
            lPos := pBlank;
          end;
        '+', '-', '*', '\', '%', '/', '<', '>', '|', '^',',':
          if lLevel = 0 then begin
            if lPos = pExpr then
              lSL.Add(Copy(aExpression,lStart,lIdx-lStart));
            lSL.AddObject(aExpression[lIdx],TObject(2));
            lPos := pBlank;
          end;
      else
        if (lLevel = 0) and (lPos = pBlank) then begin
          lStart := lIdx;
          lPos := pExpr;
        end;
      end;
      Inc(lIdx);
    end;
    if lLevel <> 0 then
      raise Exception.Create('Syntax error');
    if lPos = pExpr then
      lSL.Add(Copy(aExpression,lStart,lIdx-lStart));
    lIdx2 := lSL.IndexOfObject(TObject(2));
    if lIdx2 > -1 then begin
      lExpr := '';
      for lIdx := 0 to lIdx2 - 1 do
        if lSL.Objects[lIdx] = TObject(1) then
          lExpr := lExpr + '(' + lSL[lIdx] + ')'
        else
          lExpr := lExpr + lSL[lIdx] + ' ';
      fLeft := tExpression.Create(lExpr);
      lExpr := '';
      for lIdx := lIdx2 + 1 to lSL.Count - 1 do
        if lSL.Objects[lIdx] = TObject(1) then
          lExpr := lExpr + '(' + lSL[lIdx] + ')'
        else
          lExpr := lExpr + lSL[lIdx] + ' ';
      fRight := tExpression.Create(lExpr);
      fOperator := lSL[lIdx2];
    end else case lSL.Count of
      1:
        if Pos('(',aExpression) > 0 then begin
          Parse(lSL[0]);
        end else begin
          try
            fLeft := tValue.Create(lSL[0]);
          except
            fLeft := tVariable.Create(lSL[0]);
          end;
        end;
      2:
        begin
          fOperator := lSL[0];
          fLeft := tExpression.Create(lSL[1]);
        end;
    else
      raise Exception.Create('Syntax error: ' + aExpression);
    end;
  finally
    lSL.Free;
  end;
end;

procedure tExpression.SetVariables(aList: TStrings);
begin
  if Assigned(fLeft) then
    fLeft.SetVariables(aList);
  if Assigned(fRight) then
    fRight.SetVariables(aList);
end;

end.
